//
//  IconChooser.swift
//  Do It
//
//  Created by Jim Dovey on 2/6/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct IconChooser: View {
    @Binding var selectedIcon: String

    private struct IconSelectionInfo {
        let name: String
        let anchor: Anchor<CGRect>
    }

    private struct IconChoice: PreferenceKey {
        typealias Value = [IconSelectionInfo]
        static var defaultValue: Value = []
        static func reduce(value: inout [IconSelectionInfo],
                           nextValue: () -> [IconSelectionInfo]) {
            value.append(contentsOf: nextValue())
        }
    }
    
    private struct IconChoiceButtonStyle: ButtonStyle {
        // tertiary fill on primary background:
        //  - dark:  RGB(50,50,54)
        //  - light: RGB(239,239,240)
        @Environment(\.colorScheme) private var colorScheme
        private var backgroundColor: Color {
            switch colorScheme {
            case .dark: return Color(red: 50, green: 50, blue: 54)
            case .light: return Color(red: 239, green: 239, blue: 240)
            @unknown default:
                return Color(UIColor.tertiarySystemFill)
            }
        }
        
        private var background: some View {
            Circle().fill(Color(.tertiarySystemFill))
                .background(Circle().fill(Color(.systemBackground)))
        }
        
        func makeBody(configuration: Configuration) -> some View {
            configuration.label
                .font(.system(size: 24, weight: .bold, design: .rounded))
                .padding(6)
                .frame(width: 30)
                .padding(14)
                .background(background)
                .scaleEffect(configuration.isPressed ? 1.2 : 1)
                .padding(6)
                .hoverEffect()
                .padding(-6)
        }
    }

    var body: some View {
        VStack(spacing: 14) {
            ForEach(listIconChoices, id: \.self) { rowData in
                HStack(spacing: 14) {
                    ForEach(rowData, id: \.self) { icon in
                        Button(action: { self.selectedIcon = icon }) {
                            Image(systemName: icon)
                        }
                        .anchorPreference(key: IconChoice.self, value: .bounds) {
                            [IconSelectionInfo(name: icon, anchor: $0)]
                        }
                        .accessibility(label: Text(LocalizedStringKey(icon)))
                        .accessibility(addTraits: self.selectedIcon == icon ? .isSelected : [])
                        .accessibility(identifier: icon)
                    }
                }
            }
        }
        .accessibility(label: Text("Icon Selection"))
        .accessibility(selectionIdentifier: self.selectedIcon)
        .accessibilityElement(children: .contain)
        .buttonStyle(IconChoiceButtonStyle())
        .backgroundPreferenceValue(IconChoice.self) { values in
            GeometryReader { proxy in
                self.selectionCircle(for: values, in: proxy)
            }
        }
        .accessibilityAdjustableAction { direction in
            let allIcons = listIconChoices.flatMap { $0 }
            guard let idx = allIcons.firstIndex(of: self.selectedIcon) else {
                return
            }
            let newIdx = allIcons.index(after: idx)
            if newIdx == allIcons.endIndex {
                self.selectedIcon = allIcons[0]
            }
            else {
                self.selectedIcon = allIcons[newIdx]
            }
        }
    }

    private func selectionCircle(
        for prefs: [IconSelectionInfo],
        in proxy: GeometryProxy
    ) -> some View {
        let p = prefs.first { $0.name == selectedIcon }
        let bounds = p != nil ? proxy[p!.anchor] : .zero

        return Circle()
            .stroke(lineWidth: 3)
            .foregroundColor(Color(.opaqueSeparator))
            .frame(width: bounds.size.width + 12,
                   height: bounds.size.height + 12)
            .fixedSize()
            .offset(x: bounds.minX - 6, y: bounds.minY - 6)
    }
}

struct IconChooser_Previews: PreviewProvider {
    static var previews: some View {
        StatefulPreviewWrapper("list.bullet") {
            IconChooser(selectedIcon: $0)
        }
    }
}
